<?php

include_once get_template_directory().'/theme-includes.php';

if(!function_exists('hudson_edge_rewrite_rules_on_theme_activation')) {
    /**
     * Function that flushes rewrite rules on deactivation
     */
    function hudson_edge_rewrite_rules_on_theme_activation() {
        flush_rewrite_rules();
    }

    add_action( 'after_switch_theme', 'hudson_edge_rewrite_rules_on_theme_activation' );
}

if(!function_exists('hudson_edge_load_theme_text_domain')) {
	/**
	 * Function that sets theme domain. Hooks to after_setup_theme action
	 *
	 * @see load_theme_textdomain()
	 */
	function hudson_edge_load_theme_text_domain() {
		load_theme_textdomain( 'hudson', get_template_directory().'/languages' );
	}

	add_action('after_setup_theme', 'hudson_edge_load_theme_text_domain');
}


if (!function_exists('hudson_edge_styles')) {
	/**
	 * Function that includes theme's core styles
	 */
	function hudson_edge_styles() {
		global $hudson_edge_options;
        global $edgt_landing;
        global $edgt_toolbar;
		global $hudson_edge_icon_collections;

		//init variables
		$responsiveness = 'yes';
		$vertical_area_hidden = '';

        wp_enqueue_style( 'wp-mediaelement' );

		wp_register_style("hudson-edge-blog", HUDSON_EDGE_ROOT . "/css/blog.min.css");

		//include theme's core styles
		wp_enqueue_style("hudson-edge-default-style", HUDSON_EDGE_ROOT . "/style.css");
		wp_enqueue_style("hudson-edge-stylesheet", HUDSON_EDGE_ROOT . "/css/stylesheet.min.css");

		if(hudson_edge_load_blog_assets()) {
			wp_enqueue_style('hudson-edge-blog');
		}
		
		//define files afer which style dynamic needs to be included. It should be included last so it can override other files
		$style_dynamic_deps_array = array();
		if(hudson_edge_load_woo_assets()) {
			$style_dynamic_deps_array = array('hudson-edge-woocommerce');
		}

        if ( file_exists( HUDSON_EDGE_CSS_ROOT_DIR . '/style_dynamic.css' ) && hudson_edge_is_css_folder_writable() && ! is_multisite() ) {
            wp_enqueue_style( 'hudson-edge-style-dynamic', HUDSON_EDGE_CSS_ROOT . '/style_dynamic.css', $style_dynamic_deps_array, filemtime( HUDSON_EDGE_CSS_ROOT_DIR . '/style_dynamic.css' ) );
        } else if ( file_exists( HUDSON_EDGE_CSS_ROOT_DIR . '/style_dynamic_ms_id_' . hudson_edge_get_multisite_blog_id() . '.css' ) && hudson_edge_is_css_folder_writable() && is_multisite() ) {
            wp_enqueue_style( 'hudson-edge-style-dynamic', HUDSON_EDGE_CSS_ROOT . '/style_dynamic_ms_id_' . hudson_edge_get_multisite_blog_id() . '.css', $style_dynamic_deps_array, filemtime( HUDSON_EDGE_CSS_ROOT_DIR . '/style_dynamic_ms_id_' . hudson_edge_get_multisite_blog_id() . '.css' ) );
        } else {
            wp_enqueue_style( 'hudson-edge-style-dynamic', HUDSON_EDGE_CSS_ROOT . '/style_dynamic_callback.php', $style_dynamic_deps_array ); // Temporary case for Major update
        }

		//include icon collections styles
		if(is_array($hudson_edge_icon_collections->iconCollections) && count($hudson_edge_icon_collections->iconCollections)) {
			foreach ($hudson_edge_icon_collections->iconCollections as $collection_key => $collection_obj) {
				wp_enqueue_style('hudson-edge-'.$collection_key, $collection_obj->styleUrl);
			}
		}

		//does responsive option exists?
		if (isset($hudson_edge_options['responsiveness'])) {
			$responsiveness = $hudson_edge_options['responsiveness'];
		}

		//is responsive option turned on?
		if ($responsiveness != "no") {
			//include proper styles
			wp_enqueue_style("hudson-edge-responsive", HUDSON_EDGE_ROOT . "/css/responsive.min.css");

            if (file_exists(HUDSON_EDGE_CSS_ROOT_DIR . '/style_dynamic_responsive.css') && hudson_edge_is_css_folder_writable() && !is_multisite()) {
                wp_enqueue_style('hudson-edge-style-dynamic-responsive', HUDSON_EDGE_CSS_ROOT . "/style_dynamic_responsive.css", array(), filemtime(HUDSON_EDGE_CSS_ROOT_DIR . '/style_dynamic_responsive.css'));
            } else if (file_exists(HUDSON_EDGE_CSS_ROOT_DIR . '/style_dynamic_responsive_ms_id_' . hudson_edge_get_multisite_blog_id() . '.css') && hudson_edge_is_css_folder_writable() && is_multisite()) {
                wp_enqueue_style('hudson-edge-style-dynamic-responsive', HUDSON_EDGE_CSS_ROOT . '/style_dynamic_responsive_ms_id_' . hudson_edge_get_multisite_blog_id() . '.css', array(), filemtime(HUDSON_EDGE_CSS_ROOT_DIR . '/style_dynamic_responsive_ms_id_' . hudson_edge_get_multisite_blog_id() . '.css'));
            } else {
                wp_enqueue_style('hudson-edge-style-dynamic-responsive', HUDSON_EDGE_CSS_ROOT . "/style_dynamic_responsive_callback.php");
            }
		}
		
		//is hidden menu enabled?
		if (isset($hudson_edge_options['vertical_area_type'])){
			$vertical_area_hidden = $hudson_edge_options['vertical_area_type'];
		}

		//is left menu activated and is responsive turned on?
		if(hudson_edge_is_side_header() && $responsiveness != "no" && $vertical_area_hidden!='hidden'){
			wp_enqueue_style("hudson-edge-vertical-responsive", HUDSON_EDGE_ROOT . "/css/vertical_responsive.min.css");
		}

        //is landing turned on?
        if (isset($edgt_landing)) {
            //include ladnding specific styles
            wp_enqueue_style("hudson-edge-landing-fancybox", get_home_url() . "/demo-files/landing/css/jquery.fancybox.css");
            wp_enqueue_style("hudson-edge-landing", get_home_url() . "/demo-files/landing/css/landing_stylesheet_stripped.css");

        }

		//is toolbar turned on?
		if (isset($edgt_toolbar)) {
			//include toolbar specific styles
			wp_enqueue_style("hudson-edge-toolbar", get_home_url() . "/demo-files/toolbar/css/toolbar.css");

		}

		//include Visual Composer styles
		if (class_exists('WPBakeryVisualComposerAbstract')) {
			wp_enqueue_style( 'js_composer_front' );
		}


        $custom_css = $hudson_edge_options['custom_css'];

        if ( ! empty( $custom_css ) ) {
            wp_add_inline_style( 'hudson-edge-style-dynamic', $custom_css );
        }
	}

	add_action('wp_enqueue_scripts', 'hudson_edge_styles');
}


if(!function_exists('hudson_edge_browser_specific_styles')) {
	/**
	 * Function that includes browser specific styles. Works for Chrome on Mac and for webkit browsers
	 */
	function hudson_edge_browser_specific_styles() {
		global $is_chrome;
		global $is_safari;

		//check Chrome version
		preg_match( "#Chrome/(.+?)\.#", getenv('HTTP_USER_AGENT'), $match );
		if(!empty($match)) {
			$chrome_version = $match[1];
		} else{
			$chrome_version = 0;
		}

		//is Mac OS X?
		$mac_os = strpos(getenv('HTTP_USER_AGENT'), "Macintosh; Intel Mac OS X");

		//is Chrome on Mac with version greater than 21
		if($is_chrome && ($mac_os !== false) && ($chrome_version > 21)) {
			//include mac specific styles
			wp_enqueue_style("hudson-edge-mac-stylesheet", HUDSON_EDGE_ROOT . "/css/mac_stylesheet.css");
		}

		//is Chrome or Safari?
		if($is_chrome || $is_safari) {
			//include style for webkit browsers only
			wp_enqueue_style("hudson-edge-webkit", HUDSON_EDGE_ROOT . "/css/webkit_stylesheet.css");
		}
	}

	add_action('wp_enqueue_scripts', 'hudson_edge_browser_specific_styles');
}

/* Page ID */

if(!function_exists('hudson_edge_init_page_id')) {
	/**
	 * Function that initializes global variable that holds current page id
	 */
	function hudson_edge_init_page_id() {
		global $wp_query;
		global $edgt_page_id;

		$edgt_page_id = $wp_query->get_queried_object_id();
	}

	add_action('get_header', 'hudson_edge_init_page_id');
}


if(!function_exists('hudson_edge_google_fonts_styles')) {
    /**
     * Function that includes google fonts defined anywhere in the theme
     */
    function hudson_edge_google_fonts_styles() {
        global $hudson_edge_options;

        $font_weight_str = '100,100italic,200,200italic,300,300italic,400,400italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic';
        $font_subset_str = 'latin,latin-ext';

        //default fonts
        $default_font_family = array(
            'Open Sans',
            'Merriweather',
            'Raleway'
        );

        $modified_default_font_family = array();
        foreach ( $default_font_family as $default_font ) {
            $modified_default_font_family[] = $default_font . ':' . str_replace( ' ', '', $font_weight_str );
        }

        $default_font_string = implode( '|', $modified_default_font_family );

        $available_font_options = array(
            'google_fonts',
            'menu_google_fonts',
            'dropdown_google_fonts',
            'dropdown_wide_google_fonts',
            'dropdown_google_fonts_thirdlvl',
            'fixed_google_fonts',
            'sticky_google_fonts',
            'mobile_google_fonts',
            'h1_google_fonts',
            'h2_google_fonts',
            'h3_google_fonts',
            'h4_google_fonts',
            'h5_google_fonts',
            'h6_google_fonts',
            'text_google_fonts',
            'blockquote_font_family',
            'page_title_google_fonts',
            'page_subtitle_google_fonts',
            'page_breadcrumb_google_fonts',
            'contact_form_heading_google_fonts',
            'contact_form_section_title_google_fonts',
            'contact_form_section_subtitle_google_fonts',
            'pricing_tables_active_text_font_family',
            'pricing_tables_title_font_family',
            'pricing_tables_period_font_family',
            'pricing_tables_price_font_family',
            'pricing_tables_currency_font_family',
            'pricing_tables_button_font_family',
            'pricing_tables_content_font_family',
            'service_tables_active_text_font_family',
            'service_tables_title_font_family',
            'service_tables_content_font_family',
            'separators_with_text_text_google_fonts',
            'message_title_google_fonts',
            'counters_font_family',
            'counters_title_font_family',
            'progress_bar_horizontal_font_family',
            'progress_bar_horizontal_percentage_font_family',
            'progress_bar_vertical_font_family',
            'progress_bar_vertical_percentage_font_family',
            'list_google_fonts',
            'list_ordered_google_fonts',
            'pagination_font_family',
            'button_title_google_fonts',
            'testimonials_title_font_family',
            'testimonials_text_font_family',
            'testimonials_author_font_family',
            'testimonials_author_job_position_font_family',
            'back_to_top_text_fontfamily',
            'tabs_nav_font_family',
            'tags_font_family',
            'team_font_family',
            'footer_top_text_font_family',
            'footer_top_link_font_family',
            'footer_bottom_text_font_family',
            'footer_bottom_link_font_family',
            'footer_title_font_family',
            'sidebar_title_font_family',
            'sidebar_link_font_family',
            'sidebar_product_title_font_family',
            'side_area_title_google_fonts',
            'sidearea_link_font_family',
            'sidebar_search_text_font_family',
            'vertical_menu_google_fonts',
            'vertical_dropdown_google_fonts',
            'vertical_dropdown_google_fonts_thirdlvl',
            'popup_menu_google_fonts',
            'popup_menu_google_fonts_2nd',
            'popup_menu_3rd_google_fonts',
            'vertical_transparent_menu_google_fonts',
            'vertical_transparent_dropdown_google_fonts',
            'vertical_transparent_dropdown_google_fonts_thirdlvl',
            'popup_menu_3rd_font_family',
            'portfolio_single_big_title_font_family',
            'portfolio_single_small_title_font_family',
            'portfolio_single_meta_title_font_family',
            'top_header_text_font_family',
            'portfolio_filter_title_font_family',
            'portfolio_filter_font_family',
            'portfolio_title_standard_list_font_family',
            'portfolio_title_hover_box_list_font_family',
            'portfolio_category_standard_list_font_family',
            'portfolio_category_hover_box_list_font_family',
            'portfolio_title_list_font_family',
            'portfolio_category_list_font_family',
            'expandable_label_font_family',
            '404_title_font_family',
            '404_text_font_family',
            'woo_products_category_font_family',
            'woo_products_title_font_family',
            'woo_products_price_font_family',
            'woo_products_sale_font_family',
            'woo_products_out_of_stock_font_family',
            'woo_products_sorting_result_font_family',
            'woo_products_list_add_to_cart_font_family',
            'woo_product_single_meta_title_font_family',
            'woo_product_single_meta_info_font_family',
            'woo_product_single_title_font_family',
            'woo_products_single_add_to_cart_font_family',
            'woo_product_single_price_font_family',
            'woo_product_single_related_font_family',
            'woo_product_single_tabs_font_family',
            'woo_products_title_font_family',
            'woo_products_price_font_family',
            'content_menu_text_google_fonts',
            'blog_title_author_centered_title_google_fonts',
            'blog_title_author_centered_info_google_fonts',
            'blog_title_author_centered_author_google_fonts',
            'blog_title_author_centered_ql_title_google_fonts',
            'blog_title_author_centered_ql_info_google_fonts',
            'blog_title_author_centered_ql_author_google_fonts',
            'blog_masonry_filter_title_font_family',
            'blog_masonry_filter_font_family',
            'blog_masonry_title_google_fonts',
            'blog_masonry_info_google_fonts',
            'blog_masonry_ql_title_google_fonts',
            'blog_masonry_ql_info_google_fonts',
            'blog_masonry_ql_author_google_fonts',
            'blog_standard_type_title_google_fonts',
            'blog_standard_type_info_google_fonts',
            'blog_standard_type_ql_title_google_fonts',
            'blog_standard_type_ql_info_google_fonts',
            'blog_standard_type_ql_author_google_fonts',
            'blog_single_post_author_info_title_font_family',
            'blog_single_post_author_info_text_font_family',
            'blog_list_sections_title_font_family',
            'blog_list_sections_post_info_font_family',
            'blog_list_sections_date_font_family',
            'search_text_google_fonts',
            'side_area_text_google_fonts',
            'cf7_custom_style_1_element_font_family',
            'cf7_custom_style_1_button_font_family',
            'cf7_custom_style_2_element_font_family',
            'cf7_custom_style_2_button_font_family',
            'cf7_custom_style_3_element_font_family',
            'cf7_custom_style_3_button_font_family',
            'vc_grid_button_title_google_fonts',
            'vc_grid_load_more_button_title_google_fonts',
            'vc_grid_portfolio_filter_font_family',
            'navigation_number_font_font_family'
        );

        //define available font options array
        $fonts_array = array();
        foreach($available_font_options as $font_option) {
            //is font set and not set to default and not empty?
            if(isset($hudson_edge_options[$font_option]) && $hudson_edge_options[$font_option] !== '-1' && $hudson_edge_options[$font_option] !== '' && !hudson_edge_is_native_font($hudson_edge_options[$font_option])) {
                $font_option_string = $hudson_edge_options[$font_option].':'.$font_weight_str;
                if(!in_array($font_option_string, $fonts_array)) {
                    $fonts_array[] = $font_option_string;
                }
            }
        }

        //add google fonts set in slider
        $args = array( 'post_type' => 'slides', 'posts_per_page' => -1);
        $loop = new WP_Query( $args );

        //for each slide defined
        while ( $loop->have_posts() ) : $loop->the_post();

            //is font family for title option chosen?
            if(get_post_meta(get_the_ID(), "edgt_slide-title-font-family", true) != "") {
                $slide_title_font_string = get_post_meta(get_the_ID(), "edgt_slide-title-font-family", true) . ":".$font_weight_str;
                if(!in_array($slide_title_font_string, $fonts_array)) {
                    //include that font
                    array_push($fonts_array, $slide_title_font_string);
                }
            }

            //is font family defined for slide's text?
            if(get_post_meta(get_the_ID(), "edgt_slide-text-font-family", true) != "") {
                $slide_text_font_string = get_post_meta(get_the_ID(), "edgt_slide-text-font-family", true) . ":".$font_weight_str;
                if(!in_array($slide_text_font_string, $fonts_array)) {
                    //include that font
                    array_push($fonts_array, $slide_text_font_string);
                }
            }

            //is font family defined for slide's subtitle?
            if(get_post_meta(get_the_ID(), "edgt_slide-subtitle-font-family", true) != "") {
                $slide_subtitle_font_string = get_post_meta(get_the_ID(), "edgt_slide-subtitle-font-family", true) .":".$font_weight_str;
                if(!in_array($slide_subtitle_font_string, $fonts_array)) {
                    //include that font
                    array_push($fonts_array, $slide_subtitle_font_string);
                }
            }
        endwhile;

        wp_reset_postdata();

        if($hudson_edge_options['additional_google_fonts'] == 'yes'){

            if($hudson_edge_options['additional_google_font1'] !== '-1'){
                array_push($fonts_array, $hudson_edge_options['additional_google_font1'].":".$font_weight_str);
            }
            if($hudson_edge_options['additional_google_font2'] !== '-1'){
                array_push($fonts_array, $hudson_edge_options['additional_google_font2'].":".$font_weight_str);
            }
            if($hudson_edge_options['additional_google_font3'] !== '-1'){
                array_push($fonts_array, $hudson_edge_options['additional_google_font3'].":".$font_weight_str);
            }
            if($hudson_edge_options['additional_google_font4'] !== '-1'){
                array_push($fonts_array, $hudson_edge_options['additional_google_font4'].":".$font_weight_str);
            }
            if($hudson_edge_options['additional_google_font5'] !== '-1'){
                array_push($fonts_array, $hudson_edge_options['additional_google_font5'].":".$font_weight_str);
            }
        }

        $google_fonts_string = implode( '|', $fonts_array );

        $protocol = is_ssl() ? 'https:' : 'http:';

        //is google font option checked anywhere in theme?
        if ( count( $fonts_array ) > 0 ) {

            //include all checked fonts
            $fonts_full_list      = $default_font_string . '|' . str_replace( '+', ' ', $google_fonts_string );
            $fonts_full_list_args = array(
                'family' => urlencode( $fonts_full_list ),
                'subset' => urlencode( $font_subset_str ),
            );

            $hudson_global_fonts = add_query_arg( $fonts_full_list_args, $protocol . '//fonts.googleapis.com/css' );
            wp_enqueue_style( 'hudson-edge-google-fonts', esc_url_raw( $hudson_global_fonts ), array(), '1.0.0' );

        } else {
            //include default google font that theme is using
            $default_fonts_args          = array(
                'family' => urlencode( $default_font_string ),
                'subset' => urlencode( $font_subset_str ),
            );
            $hudson_global_fonts = add_query_arg( $default_fonts_args, $protocol . '//fonts.googleapis.com/css' );
            wp_enqueue_style( 'hudson-edge-google-fonts', esc_url_raw( $hudson_global_fonts ), array(), '1.0.0' );
        }

    }

    add_action('wp_enqueue_scripts', 'hudson_edge_google_fonts_styles');
}


if (!function_exists('hudson_edge_scripts')) {
	/**
	 * Function that includes all necessary scripts
	 */
	function hudson_edge_scripts() {
		global $hudson_edge_options;
        global $edgt_toolbar;
		global $wp_scripts;

		//init variables
		$smooth_scroll 	= true;
		$has_ajax 		= false;
		$edgt_animation = "";

		//is smooth scroll option turned on?
		if(isset($hudson_edge_options['smooth_scroll']) && $hudson_edge_options['smooth_scroll'] == "no"){
			$smooth_scroll = false;
		}

		//init theme core scripts
        wp_enqueue_script( 'jquery-ui-core' );
        wp_enqueue_script( 'jquery-ui-widget' );
        wp_enqueue_script( 'jquery-ui-mouse' );
        wp_enqueue_script( 'jquery-ui-draggable' );
        wp_enqueue_script( 'jquery-ui-droppable' );
        wp_enqueue_script( 'jquery-ui-resizable' );
        wp_enqueue_script( 'jquery-ui-selectable' );
        wp_enqueue_script( 'jquery-ui-sortable' );
        wp_enqueue_script( 'jquery-ui-accordion' );
        wp_enqueue_script( 'jquery-ui-autocomplete' );
        wp_enqueue_script( 'jquery-ui-button' );
        wp_enqueue_script( 'jquery-ui-datepicker' );
        wp_enqueue_script( 'jquery-ui-dialog' );
        wp_enqueue_script( 'jquery-effects-core' );
        wp_enqueue_script( 'jquery-effects-blind' );
        wp_enqueue_script( 'jquery-effects-bounce' );
        wp_enqueue_script( 'jquery-effects-clip' );
        wp_enqueue_script( 'jquery-effects-drop' );
        wp_enqueue_script( 'jquery-effects-explode' );
        wp_enqueue_script( 'jquery-effects-fade' );
        wp_enqueue_script( 'jquery-effects-fold' );
        wp_enqueue_script( 'jquery-effects-highlight' );
        wp_enqueue_script( 'jquery-effects-pulsate' );
        wp_enqueue_script( 'jquery-effects-scale' );
        wp_enqueue_script( 'jquery-effects-shake' );
        wp_enqueue_script( 'jquery-effects-slide' );
        wp_enqueue_script( 'jquery-effects-transfer' );
        wp_enqueue_script( 'jquery-ui-menu' );
        wp_enqueue_script( 'jquery-ui-position' );
        wp_enqueue_script( 'jquery-ui-progressbar' );
        wp_enqueue_script( 'jquery-ui-slider' );
        wp_enqueue_script( 'jquery-ui-spinner' );
        wp_enqueue_script( 'jquery-ui-tabs' );
        wp_enqueue_script( 'jquery-ui-tooltip' );
        wp_enqueue_script( 'jquery-form' );
        wp_enqueue_script( 'wp-mediaelement' );

        wp_enqueue_script("doubletaptogo", HUDSON_EDGE_ROOT."/js/plugins/doubletaptogo.min.js",array('jquery'),false,true);
        wp_enqueue_script("modernizr", HUDSON_EDGE_ROOT."/js/plugins/modernizr.min.js",array('jquery'),false,true);
        wp_enqueue_script("appear", HUDSON_EDGE_ROOT."/js/plugins/jquery.appear.min.js",array(),false,true);
        wp_enqueue_script("hoverIntent");
        wp_enqueue_script("absoluteCounter", HUDSON_EDGE_ROOT."/js/plugins/jquery.absolutecounter.min.js",array('jquery'),false,true);
        wp_enqueue_script("easyPieChart", HUDSON_EDGE_ROOT."/js/plugins/jquery.easyPieChart.min.js",array('jquery'),false,true);
        wp_enqueue_script("mixitup", HUDSON_EDGE_ROOT."/js/plugins/jquery.mixitup.min.js",array('jquery'),false,true);
        wp_enqueue_script("nicescroll", HUDSON_EDGE_ROOT."/js/plugins/jquery.nicescroll.min.js",array('jquery'),false,true);
        wp_enqueue_script("prettyPhoto", HUDSON_EDGE_ROOT."/js/plugins/jquery.prettyPhoto.min.js",array('jquery'),false,true);
        wp_enqueue_script("fitVids", HUDSON_EDGE_ROOT."/js/plugins/jquery.fitvids.min.js",array('jquery'),false,true);
        wp_enqueue_script("flexSlider", HUDSON_EDGE_ROOT."/js/plugins/jquery.flexslider-min.js",array(),false,true);
        wp_enqueue_script("infiniteScroll", HUDSON_EDGE_ROOT."/js/plugins/jquery.infiniteScroll.min.js",array('jquery'),false,true);
        wp_enqueue_script("waitforimages", HUDSON_EDGE_ROOT."/js/plugins/jquery.waitforimages.min.js",array('jquery'),false,true);
        wp_enqueue_script("waypoints", HUDSON_EDGE_ROOT."/js/plugins/waypoints.min.js",array('jquery'),false,true);
        wp_enqueue_script("jplayer", HUDSON_EDGE_ROOT."/js/plugins/jplayer.min.js",array('jquery'),false,true);
        wp_enqueue_script("bootstrapCarousel", HUDSON_EDGE_ROOT."/js/plugins/bootstrap.carousel.js",array('jquery'),false,true);
        wp_enqueue_script("skrollr", HUDSON_EDGE_ROOT."/js/plugins/skrollr.min.js",array('jquery'),false,true);
        wp_enqueue_script("Chart", HUDSON_EDGE_ROOT."/js/plugins/Chart.min.js",array('jquery'),false,true);
        wp_enqueue_script("easing", HUDSON_EDGE_ROOT."/js/plugins/jquery.easing.min.js",array('jquery'),false,true);
        wp_enqueue_script("countdown", HUDSON_EDGE_ROOT."/js/plugins/jquery.countdown.min.js",array('jquery'),false,true);
        wp_enqueue_script("multiscroll", HUDSON_EDGE_ROOT."/js/plugins/jquery.multiscroll.min.js",array('jquery'),false,true);


        wp_enqueue_script("carouFredSel", HUDSON_EDGE_ROOT."/js/plugins/jquery.carouFredSel.js",array('jquery'),false,true);
		wp_enqueue_script("one_page_scroll", HUDSON_EDGE_ROOT."/js/plugins/jquery.fullPage.min.js",array('jquery'),false,true);
		wp_enqueue_script("lemmonSlider", HUDSON_EDGE_ROOT."/js/plugins/lemmon-slider.js",array('jquery'),false,true);
		wp_enqueue_script("mousewheel", HUDSON_EDGE_ROOT."/js/plugins/jquery.mousewheel.min.js",array('jquery'),false,true);
		wp_enqueue_script("touchSwipe", HUDSON_EDGE_ROOT."/js/plugins/jquery.touchSwipe.min.js",array('jquery'),false,true);
		wp_enqueue_script("isotope", HUDSON_EDGE_ROOT."/js/plugins/jquery.isotope.min.js",array('jquery'),false,true);

	   //include google map api script
		if ($hudson_edge_options['google_maps_api_key'] != '' && isset($hudson_edge_options['google_maps_api_key']) ) {
			$google_maps_api_key = $hudson_edge_options['google_maps_api_key'];
			wp_enqueue_script('google-map-api', '//maps.googleapis.com/maps/api/js?key=' . $google_maps_api_key, array('jquery'), false, true);
		}

        if ( file_exists( HUDSON_EDGE_JS_ROOT_DIR . '/default_dynamic.js' ) && hudson_edge_is_js_folder_writable() && ! is_multisite() ) {
            wp_enqueue_script( 'hudson-edge-default-dynamic', HUDSON_EDGE_JS_ROOT . '/default_dynamic.js', array('jquery'), filemtime( HUDSON_EDGE_JS_ROOT_DIR . '/default_dynamic.js' ), true );
        } else if ( file_exists( HUDSON_EDGE_JS_ROOT_DIR . '/default_dynamic_ms_id_' . hudson_edge_get_multisite_blog_id() . '.js' ) && hudson_edge_is_js_folder_writable() && is_multisite() ) {
            wp_enqueue_script( 'hudson-edge-default-dynamic', HUDSON_EDGE_JS_ROOT . '/default_dynamic_ms_id_' . hudson_edge_get_multisite_blog_id() . '.js', array('jquery'), filemtime( HUDSON_EDGE_JS_ROOT_DIR . '/default_dynamic_ms_id_' . hudson_edge_get_multisite_blog_id() . '.js' ), true );
        } else {
            wp_enqueue_script( 'hudson-edge-default-dynamic', HUDSON_EDGE_JS_ROOT . '/default_dynamic_callback.php', array('jquery'), false, true ); // Temporary case for Major update
        }

        wp_enqueue_script("hudson-edge-default", HUDSON_EDGE_ROOT."/js/default.min.js", array('jquery'), false, true);

		if(hudson_edge_load_blog_assets()) {
			wp_enqueue_script('hudson-edge-blog', HUDSON_EDGE_ROOT."/js/blog.min.js", array('jquery'), false, true);
		}

        $custom_js = $hudson_edge_options['custom_js'];
        if ( ! empty( $custom_js ) ) {
            wp_add_inline_script( 'hudson-edge-default', $custom_js );
        }

        //is smooth scroll enabled enabled and not Mac device?
        $mac_os = strpos(getenv('HTTP_USER_AGENT'), "Macintosh; Intel Mac OS X");
        if($smooth_scroll && $mac_os == false){
            wp_enqueue_script("TweenLite", HUDSON_EDGE_ROOT."/js/plugins/TweenLite.min.js",array('jquery'),false,true);
            wp_enqueue_script("ScrollToPlugin", HUDSON_EDGE_ROOT."/js/plugins/ScrollToPlugin.min.js",array('jquery'),false,true);
            wp_enqueue_script("smoothPageScroll", HUDSON_EDGE_ROOT."/js/plugins/smoothPageScroll.js",array('jquery'),false,true);
        }

		//include comment reply script
		$wp_scripts->add_data('comment-reply', 'group', 1 );
		if (is_singular()) {
			wp_enqueue_script( "comment-reply");
		}

		if (($hudson_edge_options['page_transitions'] != "0") && (empty($edgt_animation) || ($edgt_animation != "no"))) {
			$has_ajax = true;
		} elseif (!empty($edgt_animation) && ($edgt_animation != "no"))
			$has_ajax = true;

		if ($has_ajax) {
			wp_enqueue_script("hudson-edge-ajax", HUDSON_EDGE_ROOT."/js/ajax.min.js",array('jquery'),false,true);
		}

		//include Visual Composer script
		if (class_exists('WPBakeryVisualComposerAbstract')) {
			wp_enqueue_script( 'wpb_composer_front_js' );
		}

        //is landing enabled?
        if(isset($edgt_toolbar)) {
            wp_enqueue_script("hudson-edge-toolbar", get_home_url() . "/demo-files/toolbar/js/toolbar_default.js",array('jquery'),false,true);
        }

	}

	add_action('wp_enqueue_scripts', 'hudson_edge_scripts');
}

if(!function_exists('hudson_edge_browser_specific_scripts')) {
	/**
	 * Function that loads browser specific scripts
	 */
	function hudson_edge_browser_specific_scripts() {
		global $is_IE;

		//is ie?
		if ($is_IE) {
			wp_enqueue_script("html5", HUDSON_EDGE_ROOT."/js/html5.js",array('jquery'),false,false);
		}
	}

	add_action('wp_enqueue_scripts', 'hudson_edge_browser_specific_scripts');
}

if(!function_exists('hudson_edge_woocommerce_assets')) {
	/**
	 * Function that includes all necessary scripts for WooCommerce if installed
	 */
	function hudson_edge_woocommerce_assets() {
		global $hudson_edge_options;

		//is woocommerce installed?
		if(hudson_edge_is_woocommerce_installed()) {
			if(hudson_edge_load_woo_assets()) {
				//get woocommerce specific scripts
				wp_enqueue_script("hudson-edge-woocommerce-script", HUDSON_EDGE_ROOT . "/js/woocommerce.min.js", array('jquery'), false, true);
				wp_enqueue_script("select2", HUDSON_EDGE_ROOT . "/js/plugins/select2.min.js", array('jquery'), false, true);

				//include theme's woocommerce styles
				wp_enqueue_style("hudson-edge-woocommerce", HUDSON_EDGE_ROOT . "/css/woocommerce.min.css");

				//is responsive option turned on?
				if ($hudson_edge_options['responsiveness'] == 'yes') {
					//include theme's woocommerce responsive styles
					wp_enqueue_style("hudson-edge-woocommerce-responsive", HUDSON_EDGE_ROOT . "/css/woocommerce_responsive.min.css");
				}
			}
		}
	}

	add_action('wp_enqueue_scripts', 'hudson_edge_woocommerce_assets');
}

if (!function_exists('hudson_edge_register_menus')) {
	/**
	 * Function that registers menu locations
	 */
	function hudson_edge_register_menus() {
        global $hudson_edge_options;

        if((isset($hudson_edge_options['header_bottom_appearance']) && $hudson_edge_options['header_bottom_appearance'] != "stick_with_left_right_menu") || (hudson_edge_is_side_header())){
            //header and left menu location
            register_nav_menus(
                array(
                    'top-navigation' => esc_html__( 'Top Navigation', 'hudson')
                )
            );
        }

		//popup menu location
		register_nav_menus(
			array(
                'popup-navigation' => esc_html__( 'Fullscreen Navigation', 'hudson')
			)
		);

        if((isset($hudson_edge_options['header_bottom_appearance']) && $hudson_edge_options['header_bottom_appearance'] == "stick_with_left_right_menu") && (!hudson_edge_is_side_header())){
            //header left menu location
            register_nav_menus(
                array(
                    'left-top-navigation' => esc_html__( 'Left Top Navigation', 'hudson')
                )
            );

            //header right menu location
            register_nav_menus(
                array(
                    'right-top-navigation' => esc_html__( 'Right Top Navigation', 'hudson')
                )
            );
        }
	}

	add_action( 'after_setup_theme', 'hudson_edge_register_menus' );
}

if(!function_exists('hudson_edge_add_theme_support')) {
	/**
	 * Function that adds various features to theme. Also defines image sizes that are used in a theme
	 */
	function hudson_edge_add_theme_support() {
		//add support for feed links
		add_theme_support( 'automatic-feed-links' );

		//add support for post formats
		add_theme_support('post-formats', array('gallery', 'link', 'quote', 'video', 'audio'));

		//add theme support for post thumbnails
		add_theme_support( 'post-thumbnails' );

        //add theme support for title tag
        if(function_exists('_wp_render_title_tag')) {
            add_theme_support('title-tag');
        }

        //defined content width variable
        $GLOBALS['content_width'] = 1060;

		//define thumbnail sizes
		add_image_size( 'portfolio-square', 550, 550, true );
		add_image_size( 'portfolio-landscape', 800, 600, true );
		add_image_size( 'portfolio-portrait', 600, 800, true );
		add_image_size( 'portfolio_masonry_wide', 1000, 500, true );
		add_image_size( 'portfolio_masonry_tall', 500, 1000, true );
		add_image_size( 'portfolio_masonry_large', 1000, 1000, true );
		add_image_size( 'portfolio_masonry_with_space', 700);
		add_image_size( 'blog_image_format_link_quote', 1100, 500, true);

	}

	add_action('after_setup_theme', 'hudson_edge_add_theme_support');
}

if(!function_exists('hudson_edge_rgba_color')) {
    /**
     * Function that generates rgba part of css color property
     * @param $color string hex color
     * @param $transparency float transparency value between 0 and 1
     * @return string generated rgba string
     */
    function hudson_edge_rgba_color($color, $transparency) {
        if($color !== '' && $transparency !== '') {
            $rgba_color = '';

            $rgb_color_array = hudson_edge_hex2rgb($color);
            $rgba_color .= 'rgba('.implode(', ', $rgb_color_array).', '.$transparency.')';

            return $rgba_color;
        }
    }
}

if(!function_exists('hudson_edge_wp_title_text')) {
	/**
	 * Function that sets page's title. Hooks to wp_title filter
	 * @param $title string current page title
	 * @param $sep string title separator
	 * @return string changed title text if SEO plugins aren't installed
	 */
	function hudson_edge_wp_title_text($title, $sep) {
		global $hudson_edge_options;

		//is SEO plugin installed?
		if(hudson_edge_seo_plugin_installed()) {
			//don't do anything, seo plugin will take care of it
		} else {
			//get current post id
            $id = hudson_edge_get_page_id();
			$sep = ' | ';
			$title_prefix = get_bloginfo('name');
			$title_suffix = '';

			//is WooCommerce installed and is current page shop page?
			if(hudson_edge_is_woocommerce_installed() && hudson_edge_edge_is_woocommerce_shop()) {
				//get shop page id
				$id = hudson_edge_get_woo_shop_page_id();
			}

            //is WP 4.1 at least?
            if(function_exists('_wp_render_title_tag')) {
                //set unchanged title variable so we can use it later
                $title_array = explode($sep, $title);
                $unchanged_title = array_shift($title_array);
            }

            //pre 4.1 version of WP
            else {
                //set unchanged title variable so we can use it later
                $unchanged_title = $title;
            }

			//is edgt seo enabled?
			if(isset($hudson_edge_options['disable_edgt_seo']) && $hudson_edge_options['disable_edgt_seo'] !== 'yes') {
				//get current post seo title
				$seo_title = esc_attr(get_post_meta($id, "edgt_seo_title", true));

				//is current post seo title set?
				if($seo_title !== '') {
					$title_suffix = $seo_title;
				}
			}

			//title suffix is empty, which means that it wasn't set by edgt seo
			if(empty($title_suffix)) {
				//if current page is front page append site description, else take original title string
				$title_suffix = is_front_page() ? get_bloginfo('description') : $unchanged_title;
			}

			//concatenate title string
			$title  = $title_prefix.$sep.$title_suffix;

			//return generated title string
			return $title;
		}
	}

	add_filter('wp_title', 'hudson_edge_wp_title_text', 10, 2);
}

if(!function_exists('hudson_edge_get_page_id')) {
	/**
	 * Function that returns current page / post id.
	 * Checks if current page is woocommerce page and returns that id if it is.
	 * Checks if current page is any archive page (category, tag, date, author etc.) and returns -1 because that isn't
	 * page that is created in WP admin.
	 *
	 * @return int
	 *
	 * @version 0.1
	 *
	 * @see hudson_edge_is_woocommerce_installed()
	 * @see hudson_edge_edge_is_woocommerce_shop()
	 */
	function hudson_edge_get_page_id() {
		if(hudson_edge_is_woocommerce_installed() && hudson_edge_edge_is_woocommerce_shop()) {
			return hudson_edge_get_woo_shop_page_id();
		}

        if ( is_archive() || is_search() || is_404() || ( is_front_page() && is_home() ) ) {
            return - 1;
        }

		return get_queried_object_id();
	}
}

if(!function_exists('hudson_edge_set_logo_sizes')) {
	/**
	 * Function that sets logo image dimensions to global edgt options array so it can be used in the theme
	 */
	function hudson_edge_set_logo_sizes() {
		global $hudson_edge_options;

		if (isset($hudson_edge_options['logo_image'])){
			//get logo image size
			$logo_image_sizes = hudson_edge_get_image_dimensions($hudson_edge_options['logo_image']);
			$hudson_edge_options['logo_width'] = 280;
			$hudson_edge_options['logo_height'] = 130;
	
			//is image width and height set?
			if(isset($logo_image_sizes['width']) && isset($logo_image_sizes['height'])) {
				//set those variables in global array
				$hudson_edge_options['logo_width'] = $logo_image_sizes['width'];
				$hudson_edge_options['logo_height'] = $logo_image_sizes['height'];
			}
		} 
		
		if (!empty($hudson_edge_options['ribbon_closed_image'])){
			//get logo image size
			$ribbon_closed_logo_image_sizes = hudson_edge_get_image_dimensions($hudson_edge_options['ribbon_closed_image']);
			$hudson_edge_options['ribbon_closed_image_width'] = 184;
			$hudson_edge_options['ribbon_closed_image_height'] = 184;
	
			//is image width and height set?
			if(isset($logo_image_sizes['width']) && isset($logo_image_sizes['height'])) {
				//set those variables in global array
				$hudson_edge_options['ribbon_closed_image_width'] = $ribbon_closed_logo_image_sizes['width'];
				$hudson_edge_options['ribbon_closed_image_height'] = $ribbon_closed_logo_image_sizes['height'];
			}
		}		
	}
	
	add_action('init', 'hudson_edge_set_logo_sizes', 0);
}


if(!function_exists('hudson_edge_is_default_wp_template')) {
	/**
	 * Function that checks if current page archive page, search, 404 or default home blog page
	 * @return bool
	 *
	 * @see is_archive()
	 * @see is_search()
	 * @see is_404()
	 * @see is_front_page()
	 * @see is_home()
	 */
	function hudson_edge_is_default_wp_template() {
		return is_archive() || is_search() || is_404() || (is_front_page() && is_home());
	}
}

if(!function_exists('hudson_edge_get_page_template_name')) {
	/**
	 * Returns current template file name without extension
	 * @return string name of current template file
	 */
	function hudson_edge_get_page_template_name() {
		$file_name = '';

		if(!hudson_edge_is_default_wp_template()) {
			$file_name_without_ext = preg_replace('/\\.[^.\\s]{3,4}$/', '', basename(get_page_template()));

			if($file_name_without_ext !== '') {
				$file_name = $file_name_without_ext;
			}
		}

		return $file_name;
	}
}

if(!function_exists('hudson_edge_is_main_menu_set')) {
    /**
     * Function that checks if any of main menu locations are set.
     * Checks whether top-navigation location is set, or left-top-navigation and right-top-navigation is set
     * @return bool
     *
     * @version 0.1
     */
    function hudson_edge_is_main_menu_set() {
        $has_top_nav = has_nav_menu('top-navigation');
        $has_divided_nav = has_nav_menu('left-top-navigation') && has_nav_menu('right-top-navigation');

        return $has_top_nav || $has_divided_nav;
    }
}

if(!function_exists('hudson_edge_has_shortcode')) {
	/**
	 * Function that checks whether shortcode exists on current page / post
	 * @param string shortcode to find
	 * @param string content to check. If isn't passed current post content will be used
	 * @return bool whether content has shortcode or not
	 */
	function hudson_edge_has_shortcode($shortcode, $content = '')
	{
		$has_shortcode = false;

		if ($shortcode) {
			//if content variable isn't past
			if ($content == '') {
				//take content from current post
				$page_id = hudson_edge_get_page_id();
				if (!empty($page_id)) {
					$current_post = get_post($page_id);

					if (is_object($current_post) && property_exists($current_post, 'post_content')) {
						$content = $current_post->post_content;
					}

				}
			}

			//does content has shortcode added?
			if (stripos($content, '[' . $shortcode) !== false) {
				$has_shortcode = true;
			}
		}

		return $has_shortcode;
	}
}



if(!function_exists('hudson_edge_localize_no_ajax_pages')) {
    /**
     * Function that outputs no_ajax_obj javascript variable that is used default_dynamic.php.
     * It is used for no ajax pages functionality
     *
     * Function hooks to wp_enqueue_scripts and uses wp_localize_script
     *
     * @see http://codex.wordpress.org/Function_Reference/wp_localize_script
     *
     * @uses edgt_get_posts_without_ajax()
     * @uses edgt_get_pages_without_ajax()
     * @uses hudson_edge_get_wpml_pages_for_current_page()
     * @uses hudson_edge_get_woocommerce_pages()
     *
     * @version 0.1
     */
    function hudson_edge_localize_no_ajax_pages() {
        global $hudson_edge_options;

        //is ajax enabled?
        if(hudson_edge_is_ajax_enabled()) {
            $no_ajax_pages = array();

            //get posts that have ajax disabled and merge with main array
            $no_ajax_pages = array_merge($no_ajax_pages, hudson_edge_get_objects_without_ajax());

            //is wpml installed?
            if(hudson_edge_is_wpml_installed()) {
                //get translation pages for current page and merge with main array
                $no_ajax_pages = array_merge($no_ajax_pages, hudson_edge_get_wpml_pages_for_current_page());
            }

            //is woocommerce installed?
            if(hudson_edge_is_woocommerce_installed()) {
                //get all woocommerce pages and products and merge with main array
                $no_ajax_pages = array_merge($no_ajax_pages, hudson_edge_get_woocommerce_pages());
            }

            //do we have some internal pages that won't to be without ajax?
            if (isset($hudson_edge_options['internal_no_ajax_links'])) {
                //get array of those pages
                $options_no_ajax_pages_array = explode(',', $hudson_edge_options['internal_no_ajax_links']);

                if(is_array($options_no_ajax_pages_array) && count($options_no_ajax_pages_array)) {
                    $no_ajax_pages = array_merge($no_ajax_pages, $options_no_ajax_pages_array);
                }
            }

            //add logout url to main array
            $no_ajax_pages[] = wp_specialchars_decode(wp_logout_url());

            //finally localize script so we can use it in default_dynamic
            wp_localize_script( 'edgt_default_dynamic', 'no_ajax_obj', array(
                'no_ajax_pages' => $no_ajax_pages
            ));
        }
    }

    add_action('wp_enqueue_scripts', 'hudson_edge_localize_no_ajax_pages');
}

if(!function_exists('hudson_edge_get_objects_without_ajax')) {
    /**
     * Function that returns urls of objects that have ajax disabled.
     * Works for posts, pages and portfolio pages.
     * @return array array of urls of posts that have ajax disabled
     *
     * @version 0.1
     */
    function hudson_edge_get_objects_without_ajax() {
        $posts_without_ajax = array();

        $posts_args =  array(
            'post_type'  => array('post', 'portfolio_page', 'page'),
            'post_status' => 'publish',
            'meta_key' => 'edgt_show-animation',
            'meta_value' => 'no_animation'
        );

        $posts_query = new WP_Query($posts_args);

        if($posts_query->have_posts()) {
            while($posts_query->have_posts()) {
                $posts_query->the_post();
                $posts_without_ajax[] = get_permalink(get_the_ID());
            }
        }

        wp_reset_postdata();

        return $posts_without_ajax;
    }
}

if(!function_exists('hudson_edge_is_ajax_enabled')) {
    /**
     * Function that checks if ajax is enabled.
     * @return bool
     *
     * @version 0.1
     */
    function hudson_edge_is_ajax_enabled() {
        global $hudson_edge_options;

        $has_ajax = false;

        if(isset($hudson_edge_options['page_transitions']) && $hudson_edge_options['page_transitions'] !== '0') {
            $has_ajax = true;
        }

        return $has_ajax;
    }
}

if(!function_exists('hudson_edge_is_ajax_header_animation_enabled')) {
    /**
     * Function that checks if header animation with ajax is enabled.
     * @return boolean
     *
     * @version 0.1
     */
    function hudson_edge_is_ajax_header_animation_enabled() {
        global $hudson_edge_options;

        $has_header_animation = false;

        if(isset($hudson_edge_options['page_transitions']) && $hudson_edge_options['page_transitions'] !== '0' && isset($hudson_edge_options['ajax_animate_header']) && $hudson_edge_options['ajax_animate_header'] == 'yes') {
            $has_header_animation = true;
        }

        return $has_header_animation;
    }
}

if(!function_exists('hudson_edge_horizontal_slider_icon_classes')) {
	/**
	 * Returns classes for left and right arrow for sliders
	 *
	 * @param $icon_class
	 * @return array
	 */
	function hudson_edge_horizontal_slider_icon_classes($icon_class) {

		switch($icon_class) {
			case 'arrow_carrot-left_alt2':
				$left_icon_class = 'arrow_carrot-left_alt2';
				$right_icon_class = 'arrow_carrot-right_alt2';
				break;
			case 'arrow_carrot-2left_alt2':
				$left_icon_class = 'arrow_carrot-2left_alt2';
				$right_icon_class = 'arrow_carrot-2right_alt2';
				break;
			case 'arrow_triangle-left_alt2':
				$left_icon_class = 'arrow_triangle-left_alt2';
				$right_icon_class = 'arrow_triangle-right_alt2';
				break;
			case 'icon-arrows-drag-left-dashed':
				$left_icon_class = 'icon-arrows-drag-left-dashed';
				$right_icon_class = 'icon-arrows-drag-right-dashed';
				break;
			case 'icon-arrows-drag-left-dashed':
				$left_icon_class = 'icon-arrows-drag-left-dashed';
				$right_icon_class = 'icon-arrows-drag-right-dashed';
				break;
			case 'icon-arrows-left-double-32':
				$left_icon_class = 'icon-arrows-left-double-32';
				$right_icon_class = 'icon-arrows-right-double';
				break;
			case 'icon-arrows-slide-left1':
				$left_icon_class = 'icon-arrows-slide-left1';
				$right_icon_class = 'icon-arrows-slide-right1';
				break;
			case 'icon-arrows-slide-left2':
				$left_icon_class = 'icon-arrows-slide-left2';
				$right_icon_class = 'icon-arrows-slide-right2';
				break;
			case 'icon-arrows-slim-left-dashed':
				$left_icon_class = 'icon-arrows-slim-left-dashed';
				$right_icon_class = 'icon-arrows-slim-right-dashed';
				break;
			case 'ion-arrow-left-a':
				$left_icon_class = 'ion-arrow-left-a';
				$right_icon_class = 'ion-arrow-right-a';
				break;
			case 'ion-arrow-left-b':
				$left_icon_class = 'ion-arrow-left-b';
				$right_icon_class = 'ion-arrow-right-b';
				break;
			case 'ion-arrow-left-c':
				$left_icon_class = 'ion-arrow-left-c';
				$right_icon_class = 'ion-arrow-right-c';
				break;
			case 'ion-ios-arrow-':
				$left_icon_class = $icon_class.'back';
				$right_icon_class = $icon_class.'forward';
				break;
			case 'ion-ios-fastforward':
				$left_icon_class = 'ion-ios-rewind';
				$right_icon_class = 'ion-ios-fastforward';
				break;
			case 'ion-ios-fastforward-outline':
				$left_icon_class = 'ion-ios-rewind-outline';
				$right_icon_class = 'ion-ios-fastforward-outline';
				break;
			case 'ion-ios-skipbackward':
				$left_icon_class = 'ion-ios-skipbackward';
				$right_icon_class = 'ion-ios-skipforward';
				break;
			case 'ion-ios-skipbackward-outline':
				$left_icon_class = 'ion-ios-skipbackward-outline';
				$right_icon_class = 'ion-ios-skipforward-outline';
				break;
			case 'ion-android-arrow-':
				$left_icon_class = $icon_class.'back';
				$right_icon_class = $icon_class.'forward';
				break;
			case 'ion-android-arrow-dropleft-circle':
				$left_icon_class = 'ion-android-arrow-dropleft-circle';
				$right_icon_class = 'ion-android-arrow-dropright-circle';
				break;
			default:
				$left_icon_class = $icon_class.'left';
				$right_icon_class = $icon_class.'right';
		}

		$icon_classes = array(
			'left_icon_class' => $left_icon_class,
			'right_icon_class' => $right_icon_class
		);

    	return $icon_classes;

	}

}

if(!function_exists('hudson_edge_get_side_menu_icon_html')) {
	/**
	 * Function that outputs html for side area icon opener.
	 * Uses $hudson_edge_icon_collections global variable
	 * @return string generated html
	 */
	function hudson_edge_get_side_menu_icon_html() {
		global $hudson_edge_icon_collections, $hudson_edge_options;

		$icon_html = '';

		if(isset($hudson_edge_options['side_area_button_icon_pack']) && $hudson_edge_options['side_area_button_icon_pack'] !== '') {
			$icon_pack = $hudson_edge_options['side_area_button_icon_pack'];
			if ($icon_pack !== '') {
				$icon_collection_obj = $hudson_edge_icon_collections->getIconCollection($icon_pack);
				$icon_field_name = 'side_area_icon_'. $icon_collection_obj->param;

				if(isset($hudson_edge_options[$icon_field_name]) && $hudson_edge_options[$icon_field_name] !== ''){
					$icon_single = $hudson_edge_options[$icon_field_name];

					if (method_exists($icon_collection_obj, 'render')) {
						$icon_html = $icon_collection_obj->render($icon_single);
					}
				}
			}
		}

		return $icon_html;
	}
}

if (!function_exists('hudson_edge_vc_grid_elements_enabled')) {

	/**
	 * Function that checks if Visual Composer Grid Elements are enabled
	 *
	 * @return bool
	 */
	function hudson_edge_vc_grid_elements_enabled() {

		global $hudson_edge_options;
		$vc_grid_enabled = false;

		if (isset($hudson_edge_options['enable_grid_elements']) && $hudson_edge_options['enable_grid_elements'] == 'yes') {

			$vc_grid_enabled = true;

		}

		return $vc_grid_enabled;

	}

}

if(!function_exists('hudson_edge_visual_composer_grid_elements')) {

	/**
	 * Removes Visual Composer Grid Elements post type if VC Grid option disabled
	 * and enables Visual Composer Grid Elements post type
	 * if VC Grid option enabled
	 */
	function hudson_edge_visual_composer_grid_elements() {

		if(!hudson_edge_vc_grid_elements_enabled()){

			remove_action( 'init', 'vc_grid_item_editor_create_post_type' );

		}
	}

	add_action('vc_after_init', 'hudson_edge_visual_composer_grid_elements', 12);
}

if(!function_exists('hudson_edge_grid_elements_ajax_disable')) {
	/**
	 * Function that disables ajax transitions if grid elements are enabled in theme options
	 */
	function hudson_edge_grid_elements_ajax_disable() {
		global $hudson_edge_options;

		if(hudson_edge_vc_grid_elements_enabled()) {
			$hudson_edge_options['page_transitions'] = '0';
		}
	}

	add_action('wp', 'hudson_edge_grid_elements_ajax_disable');
}


if(!function_exists('hudson_edge_get_vc_version')) {
	/**
	 * Return Visual Composer version string
	 *
	 * @return bool|string
	 */
	function hudson_edge_get_vc_version() {
		if(hudson_edge_visual_composer_installed()) {
			return WPB_VC_VERSION;
		}

		return false;
	}
}

if(!function_exists('hudson_edge_get_dynamic_sidebar')){
	/**
	 * Return Custom Widget Area content
	 *
	 * @return string
	 */
	function hudson_edge_get_dynamic_sidebar($index = 1){
		$sidebar_contents = "";
		ob_start();
		dynamic_sidebar($index);
		$sidebar_contents = ob_get_clean();
		return $sidebar_contents;
	}
}

if ( ! function_exists('hudson_edge_is_gutenberg_installed') ) {
	/**
	 * Function that checks if Gutenberg plugin installed
	 * @return bool
	 */
	function hudson_edge_is_gutenberg_installed() {
		return function_exists( 'is_gutenberg_page' ) && is_gutenberg_page();
	}
}

if ( ! function_exists('hudson_edge_is_wp_gutenberg_installed') ) {
    /**
     * Function that checks if WordPress 5.x with Gutenberg editor installed
     *
     * @return bool
     */
    function hudson_edge_is_wp_gutenberg_installed() {
        return class_exists( 'WP_Block_Type' );
    }
}

if ( ! function_exists('hudson_edge_enqueue_editor_customizer_styles') ) {
    /**
     * Enqueue supplemental block editor styles
     */
    function hudson_edge_enqueue_editor_customizer_styles() {
        wp_enqueue_style( 'hudson-editor-customizer-style', HUDSON_EDGE_CSS_ROOT . '/admin/editor-customizer-style.css' );
        wp_enqueue_style( 'hudson-editor-blocks-style', HUDSON_EDGE_CSS_ROOT . '/admin/editor-blocks-style.css' );
    }

    add_action( 'enqueue_block_editor_assets', 'hudson_edge_enqueue_editor_customizer_styles' );
}
